/*
 * Decompiled with CFR 0.152.
 */
package gg.hyservers.plugin.network;

import com.hypixel.hytale.logger.HytaleLogger;
import gg.hyservers.plugin.libs.gson.JsonObject;
import gg.hyservers.plugin.libs.gson.JsonParser;
import gg.hyservers.plugin.network.ProtocolConstants;
import gg.hyservers.plugin.rewards.RewardExecutor;
import gg.hyservers.plugin.util.HmacUtil;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.util.Arrays;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

public class NotificationServer {
    private final String host;
    private final int port;
    private final String sharedSecret;
    private final boolean debug;
    private final RewardExecutor rewardExecutor;
    private final HytaleLogger logger;
    private ServerSocket serverSocket;
    private ExecutorService executor;
    private volatile boolean running;

    public NotificationServer(String host, int port, String sharedSecret, boolean debug, RewardExecutor rewardExecutor, HytaleLogger logger) {
        this.host = host;
        this.port = port;
        this.sharedSecret = sharedSecret;
        this.debug = debug;
        this.rewardExecutor = rewardExecutor;
        this.logger = logger;
    }

    public void start() throws IOException {
        if (this.running) {
            throw new IllegalStateException("Notification server is already running");
        }
        this.serverSocket = new ServerSocket();
        this.serverSocket.setReuseAddress(true);
        this.serverSocket.bind(new InetSocketAddress(this.host, this.port));
        this.executor = Executors.newFixedThreadPool(2, r -> {
            Thread t = new Thread(r, "HyServers-Network");
            t.setDaemon(true);
            return t;
        });
        this.running = true;
        this.executor.submit(this::acceptLoop);
        ((HytaleLogger.Api)this.logger.atInfo()).log("HyServers notification server listening on %s:%d", (Object)this.host, this.port);
    }

    public void stop() {
        this.running = false;
        if (this.serverSocket != null && !this.serverSocket.isClosed()) {
            try {
                this.serverSocket.close();
            }
            catch (IOException e) {
                ((HytaleLogger.Api)((HytaleLogger.Api)this.logger.atWarning()).withCause((Throwable)e)).log("Error closing server socket");
            }
        }
        if (this.executor != null) {
            this.executor.shutdown();
            try {
                if (!this.executor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.executor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.executor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        ((HytaleLogger.Api)this.logger.atInfo()).log("HyServers notification server stopped");
    }

    private void acceptLoop() {
        while (this.running) {
            try {
                Socket client = this.serverSocket.accept();
                client.setSoTimeout(5000);
                this.executor.submit(() -> this.handleConnection(client));
            }
            catch (SocketException e) {
                if (!this.running) continue;
                ((HytaleLogger.Api)((HytaleLogger.Api)this.logger.atWarning()).withCause((Throwable)e)).log("Socket error in accept loop");
            }
            catch (IOException e) {
                if (!this.running) continue;
                ((HytaleLogger.Api)((HytaleLogger.Api)this.logger.atWarning()).withCause((Throwable)e)).log("Error accepting connection");
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void handleConnection(Socket client) {
        String clientAddr = client.getRemoteSocketAddress().toString();
        try (Socket socket = client;
             DataInputStream in = new DataInputStream(client.getInputStream());
             DataOutputStream out = new DataOutputStream(client.getOutputStream());){
            if (this.debug) {
                ((HytaleLogger.Api)this.logger.atInfo()).log("[Debug] Connection from %s", (Object)clientAddr);
            }
            byte[] header = new byte[ProtocolConstants.HEADER_LENGTH];
            in.readFully(header);
            if (!Arrays.equals(header, ProtocolConstants.MAGIC_HEADER)) {
                this.sendError(out, "Invalid protocol header");
                if (!this.debug) return;
                ((HytaleLogger.Api)this.logger.atInfo()).log("[Debug] Invalid header from %s", (Object)clientAddr);
                return;
            }
            int payloadLength = in.readUnsignedShort();
            if (payloadLength <= 0 || payloadLength > 4096) {
                this.sendError(out, "Invalid payload size: " + payloadLength);
                ((HytaleLogger.Api)this.logger.atWarning()).log("Invalid payload size %d from %s", payloadLength, (Object)clientAddr);
                return;
            }
            byte[] payloadBytes = new byte[payloadLength];
            in.readFully(payloadBytes);
            String jsonPayload = new String(payloadBytes, ProtocolConstants.PAYLOAD_CHARSET);
            if (this.debug) {
                ((HytaleLogger.Api)this.logger.atInfo()).log("[Debug] Received payload: %s", (Object)jsonPayload);
            }
            if (!HmacUtil.validate(jsonPayload, this.sharedSecret)) {
                this.sendError(out, "Invalid signature");
                ((HytaleLogger.Api)this.logger.atWarning()).log("Invalid signature from %s", (Object)clientAddr);
                return;
            }
            JsonObject json = JsonParser.parseString(jsonPayload).getAsJsonObject();
            long timestamp = json.get("timestamp").getAsLong();
            long now = System.currentTimeMillis() / 1000L;
            long age = Math.abs(now - timestamp);
            if (age > 300L) {
                this.sendError(out, "Packet expired (age: " + age + "s)");
                ((HytaleLogger.Api)this.logger.atWarning()).log("Expired packet from %s (age: %ds)", (Object)clientAddr, age);
                return;
            }
            String type = json.get("type").getAsString();
            String player = json.get("player").getAsString();
            if (player == null || player.isBlank()) {
                this.sendError(out, "Missing player name");
                return;
            }
            switch (type) {
                case "vote": {
                    JsonObject rewardObj = json.has("reward") && !json.get("reward").isJsonNull() ? json.getAsJsonObject("reward") : null;
                    this.rewardExecutor.processVote(player, rewardObj);
                    this.sendSuccess(out, "Vote processed for " + player);
                    ((HytaleLogger.Api)this.logger.atInfo()).log("Vote received for %s from %s", (Object)player, (Object)clientAddr);
                    return;
                }
                case "review": {
                    JsonObject data = json.getAsJsonObject("data");
                    int rating = data != null && data.has("rating") ? data.get("rating").getAsInt() : 0;
                    this.rewardExecutor.processReview(player, rating);
                    this.sendSuccess(out, "Review processed for " + player);
                    ((HytaleLogger.Api)this.logger.atInfo()).log("Review received for %s (rating: %d) from %s", (Object)player, (Object)rating, (Object)clientAddr);
                    return;
                }
            }
            this.sendError(out, "Unknown notification type: " + type);
            ((HytaleLogger.Api)this.logger.atWarning()).log("Unknown notification type '%s' from %s", (Object)type, (Object)clientAddr);
            return;
        }
        catch (IOException e) {
            if (!this.debug) return;
            ((HytaleLogger.Api)((HytaleLogger.Api)this.logger.atWarning()).withCause((Throwable)e)).log("I/O error handling connection from %s", (Object)clientAddr);
            return;
        }
        catch (Exception e) {
            ((HytaleLogger.Api)((HytaleLogger.Api)this.logger.atWarning()).withCause((Throwable)e)).log("Error handling connection from %s", (Object)clientAddr);
        }
    }

    private void sendSuccess(DataOutputStream out, String message) throws IOException {
        this.sendResponse(out, "{\"success\":true,\"message\":\"" + this.escapeJson(message) + "\"}");
    }

    private void sendError(DataOutputStream out, String error) throws IOException {
        this.sendResponse(out, "{\"success\":false,\"error\":\"" + this.escapeJson(error) + "\"}");
    }

    private void sendResponse(DataOutputStream out, String json) throws IOException {
        byte[] payload = json.getBytes(ProtocolConstants.PAYLOAD_CHARSET);
        out.write(ProtocolConstants.MAGIC_HEADER);
        out.writeShort(payload.length);
        out.write(payload);
        out.flush();
    }

    private String escapeJson(String s) {
        return s.replace("\\", "\\\\").replace("\"", "\\\"");
    }
}

